<?php

/**
 * Abstract Base Class for Commands
 *
 * @package _121Digital\Connect
 */

namespace _121Digital\Connect\Commands;

/**
 * Abstract base class for all command classes in the plugin.
 *
 * This class provides a structure for executing commands with arguments.
 * Child classes should implement the `execute()` method with their specific logic.
 */
abstract class CommandBase
{
    /**
     * The arguments passed to the command.
     *
     * @var array
     */
    protected array $args;

    /**
     * Constructor to initialize the command with arguments.
     *
     * @param array $args Optional associative array of arguments.
     */
    public function __construct(array $args = [])
    {
        $this->args = $args;
    }

    /**
     * Execute the command.
     *
     * Child classes must implement this method.
     *
     * @return array Result array with 'success', 'message', and 'data' keys
     */
    abstract public function execute(): array;

    /**
     * Get all arguments passed to the command.
     *
     * @return array The command arguments.
     */
    protected function getArgs(): array
    {
        return $this->args;
    }

    /**
     * Get a specific argument value
     *
     * @param string $key
     * @param mixed $default
     * @return mixed
     */
    protected function getArg(string $key, $default = null)
    {
        return $this->args[$key] ?? $default;
    }

    /**
     * Set or update the arguments for the command.
     *
     * @param array $args Associative array of new arguments.
     * @return void
     */
    public function setArgs(array $args = []): void
    {
        $this->args = $args;
    }

    /**
     * Helper method to return a success response
     *
     * @param string $message
     * @param array $data
     * @return array
     */
    protected function success(string $message, array $data = []): array
    {
        return [
            'success' => true,
            'message' => $message,
            'data' => $data,
        ];
    }

    /**
     * Helper method to return an error response
     *
     * @param string $message
     * @param array $data
     * @return array
     */
    protected function error(string $message, array $data = []): array
    {
        return [
            'success' => false,
            'message' => $message,
            'data' => $data,
        ];
    }
}
