<?php

/**
 * Matomo Integration
 *
 * @package _121Digital\Connect
 */

namespace _121Digital\Connect\Integration;

/**
 * Matomo Class
 *
 * Handles Matomo analytics tracking code integration
 */
class Matomo
{
    /**
     * Register component
     *
     * @return void
     */
    public function register(): void
    {
        add_action('wp_head', [$this, 'renderTrackingCode']);
    }

    /**
     * Check if a Matomo tracking code has been set
     *
     * @return bool
     */
    public function hasTrackingCode(): bool
    {
        return get_option('sc_matomo_tracking_code') !== false;
    }

    /**
     * Save or update the Matomo tracking code
     *
     * @param string $trackingCode The tracking code to store
     * @return void
     */
    public function setTrackingCode(string $trackingCode): void
    {
        update_option('sc_matomo_tracking_code', $trackingCode);
    }

    /**
     * Retrieve the Matomo tracking code
     *
     * @return string
     */
    public function getTrackingCode(): string
    {
        $trackingCode = get_option('sc_matomo_tracking_code');
        return $trackingCode === false ? '' : $trackingCode;
    }

    /**
     * Validate if tracking code can be safely rendered
     *
     * @return bool
     */
    public function canRenderTrackingCode(): bool
    {
        $code = $this->getTrackingCode();

        if ($code === '') {
            return false;
        }

        // Quick safety check
        $blacklist = ['<iframe', '<object', '<embed', '<link', '<meta', 'onerror=', 'onload='];
        foreach ($blacklist as $tag) {
            if (stripos($code, $tag) !== false) {
                return false;
            }
        }

        // Must contain basic Matomo function calls
        $required = ['_paq.push', 'trackPageView', 'enableLinkTracking', 'setTrackerUrl', 'setSiteId'];
        foreach ($required as $item) {
            if (stripos($code, $item) === false) {
                return false;
            }
        }

        // Check that the script URL is only to your known analytics domain
        if (!str_contains($code, 'analytics.121domains.co.uk')) {
            return false;
        }

        return true;
    }

    /**
     * Check if tracking code is empty
     *
     * @return bool
     */
    public function trackingCodeEmpty(): bool
    {
        return $this->getTrackingCode() === '';
    }

    /**
     * Render the Matomo tracking code in wp_head
     *
     * @return void
     */
    public function renderTrackingCode(): void
    {
        if ($this->canRenderTrackingCode()) {
            $trackingCode = $this->getTrackingCode();
            echo "<!-- START REG121 Tracking Code -->\n";
            if ($trackingCode !== '') {
                echo $trackingCode;
            } else {
                echo "<!-- No tracking code defined. -->\n";
            }
            echo "<!-- END REG121 Tracking Code -->\n";
        } else {
            echo "<!-- REG121 Matomo Tracking Code Blocked For Security Reasons -->\n";
        }
    }
}
