<?php
/**
 * Modern Dashboard Enhancement
 *
 * @package _121Digital\Connect
 */

namespace _121Digital\Connect\Admin;

/**
 * ModernDashboard Class
 *
 * Enhances the WordPress dashboard with modern styling and functionality
 */
class ModernDashboard
{
    /**
     * Register dashboard hooks
     *
     * @return void
     */
    public function register(): void
    {
        add_action('wp_dashboard_setup', [$this, 'addModernDashboardWidgets']);
        add_action('admin_enqueue_scripts', [$this, 'enqueueDashboardStyles']);
        add_action('admin_head', [$this, 'addDashboardCustomStyles']);
        add_action('admin_init', [$this, 'removeDefaultWordpressWelcomeWidget']);
    }

    /**
     * Add modern dashboard widgets
     *
     * @return void
     */
    public function addModernDashboardWidgets(): void
    {
        if (!$this->shouldForceUI()) {
            return;
        }

        // Remove default WordPress widgets that might cause layout issues
        remove_meta_box('dashboard_quick_press', 'dashboard', 'side');
        remove_meta_box('dashboard_recent_drafts', 'dashboard', 'side');
        remove_meta_box('dashboard_primary', 'dashboard', 'side');
        remove_meta_box('dashboard_secondary', 'dashboard', 'side');
        remove_meta_box('dashboard_incoming_links', 'dashboard', 'normal');
        remove_meta_box('dashboard_plugins', 'dashboard', 'normal');
        remove_meta_box('dashboard_right_now', 'dashboard', 'normal');
        remove_meta_box('dashboard_activity', 'dashboard', 'normal');

        // Add a modern welcome widget
        wp_add_dashboard_widget(
            'modern_welcome_widget',
            'Welcome to Your Modern Dashboard',
            [$this, 'renderWelcomeWidget'],
            'dashboard',
            'normal',
            'high'
        );

        // Add a quick actions widget
        wp_add_dashboard_widget(
            'quick_actions_widget',
            'Quick Actions',
            [$this, 'renderQuickActionsWidget'],
            'dashboard',
            'side',
            'high'
        );

        // Add a site overview widget
        wp_add_dashboard_widget(
            'site_overview_widget',
            'Site Overview',
            [$this, 'renderSiteOverviewWidget'],
            'dashboard',
            'side',
            'high'
        );

        // Add a design widget (full width)
        wp_add_dashboard_widget(
            'design_widget',
            'Design',
            [$this, 'renderDesignWidget'],
            'dashboard',
            'normal',
            'high'
        );
    }

    public function removeDefaultWordpressWelcomeWidget(): void
    {
        remove_meta_box('welcome_panel', 'dashboard', 'normal');
    }

    /**
     * Enqueue dashboard styles
     *
     * @return void
     */
    public function enqueueDashboardStyles(): void
    {
        if (!$this->shouldForceUI()) {
            return;
        }

        wp_enqueue_style(
            'modern-dashboard',
            SC_PLUGIN_URL . 'src/Admin/modern-dashboard.css',
            ['sc-admin-styles'],
            SC_PLUGIN_VERSION
        );

        wp_enqueue_script(
            'modern-dashboard',
            SC_PLUGIN_URL . 'src/Admin/modern-dashboard.js',
            ['jquery'],
            SC_PLUGIN_VERSION,
            true
        );
    }

    /**
     * Add custom dashboard styles
     *
     * @return void
     */
    public function addDashboardCustomStyles(): void
    {
        if (!$this->shouldForceUI()) {
            return;
        }
        ?>
        <style>
        /* Modern dashboard enhancements */
        .modern-dashboard-widget {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            border-radius: 12px;
            padding: 30px;
            margin: 0;
            box-shadow: 0 8px 32px rgba(102, 126, 234, 0.3);
        }

        .modern-dashboard-widget h3 {
            color: white;
            font-size: 24px;
            font-weight: 700;
            margin: 0 0 15px 0;
            letter-spacing: -0.02em;
        }

        .modern-dashboard-widget p {
            color: rgba(255, 255, 255, 0.9);
            font-size: 16px;
            line-height: 1.6;
            margin: 0 0 20px 0;
        }

        .quick-actions-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 15px;
            margin-top: 20px;
        }

        .quick-action-card {
            background: white;
            border-radius: 8px;
            padding: 20px;
            text-align: center;
            transition: all 0.2s ease;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
            border: 1px solid #e9ecef;
        }

        .quick-action-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.15);
        }

        .quick-action-card .dashicons {
            font-size: 32px;
            color: #667eea;
            margin-bottom: 10px;
        }

        .quick-action-card h4 {
            margin: 0 0 8px 0;
            color: #2c3e50;
            font-size: 14px;
            font-weight: 600;
        }

        .quick-action-card p {
            margin: 0;
            color: #6c757d;
            font-size: 12px;
            line-height: 1.4;
        }

        .site-stats-grid {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(150px, 1fr));
            gap: 15px;
            margin-top: 20px;
        }

        .stat-card {
            background: white;
            border-radius: 8px;
            padding: 20px;
            text-align: center;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
            border: 1px solid #e9ecef;
        }

        .stat-number {
            font-size: 28px;
            font-weight: 700;
            color: #667eea;
            margin: 0 0 5px 0;
        }

        .stat-label {
            font-size: 12px;
            color: #6c757d;
            text-transform: uppercase;
            letter-spacing: 0.5px;
            margin: 0;
        }

        .modern-button {
            background: rgba(255, 255, 255, 0.2);
            border: 1px solid rgba(255, 255, 255, 0.3);
            color: white;
            padding: 10px 20px;
            border-radius: 6px;
            text-decoration: none;
            font-weight: 500;
            font-size: 14px;
            transition: all 0.2s ease;
            display: inline-block;
        }

        .modern-button:hover {
            background: rgba(255, 255, 255, 0.3);
            color: white;
            text-decoration: none;
            transform: translateY(-1px);
        }
        </style>
        <?php
    }

    /**
     * Render welcome widget
     *
     * @return void
     */
    public function renderWelcomeWidget(): void
    {
        $current_user = wp_get_current_user();
        ?>
        <div class="modern-dashboard-widget">
            <h3>Welcome back, <?php echo esc_html($current_user->display_name); ?>!</h3>
            <p>Your website is looking great. Here's what you can do to keep it running smoothly.</p>
            <a href="<?php echo admin_url('post-new.php'); ?>" class="modern-button">Create New Post</a>
            <a href="<?php echo admin_url('customize.php'); ?>" class="modern-button">Customize Design</a>
        </div>
        <?php
    }

    /**
     * Render quick actions widget
     *
     * @return void
     */
    public function renderQuickActionsWidget(): void
    {
        ?>
        <div class="quick-actions-grid">
            <div class="quick-action-card">
                <span class="dashicons dashicons-edit"></span>
                <h4>Write Post</h4>
                <p>Create new content</p>
            </div>
            <div class="quick-action-card">
                <span class="dashicons dashicons-admin-page"></span>
                <h4>Add Page</h4>
                <p>Create new pages</p>
            </div>
            <div class="quick-action-card">
                <span class="dashicons dashicons-admin-appearance"></span>
                <h4>Customize</h4>
                <p>Design & styling</p>
            </div>
            <div class="quick-action-card">
                <span class="dashicons dashicons-admin-media"></span>
                <h4>Media Library</h4>
                <p>Manage files</p>
            </div>
            <div class="quick-action-card">
                <span class="dashicons dashicons-admin-users"></span>
                <h4>Users</h4>
                <p>Manage accounts</p>
            </div>
            <div class="quick-action-card">
                <span class="dashicons dashicons-admin-settings"></span>
                <h4>Settings</h4>
                <p>Site configuration</p>
            </div>
        </div>
        <?php
    }

    /**
     * Render site overview widget
     *
     * @return void
     */
    public function renderSiteOverviewWidget(): void
    {
        $post_count = wp_count_posts();
        $page_count = wp_count_posts('page');
        $comment_count = wp_count_comments();
        $user_count = count_users();
        ?>
        <div class="site-stats-grid">
            <div class="stat-card">
                <div class="stat-number"><?php echo $post_count->publish; ?></div>
                <div class="stat-label">Posts</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?php echo $page_count->publish; ?></div>
                <div class="stat-label">Pages</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?php echo $comment_count->approved; ?></div>
                <div class="stat-label">Comments</div>
            </div>
            <div class="stat-card">
                <div class="stat-number"><?php echo $user_count['total_users']; ?></div>
                <div class="stat-label">Users</div>
            </div>
        </div>
        <?php
    }

    /**
     * Render design widget
     *
     * @return void
     */
    public function renderDesignWidget(): void
    {
        ?>
        <div class="design-widget">
            <p style="color: #6c757d; margin-bottom: 20px; font-size: 16px;">Customize the appearance of your website using the block editor.</p>
            
            <div class="design-grid">
                <div class="design-card">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-menu" style="font-size: 24px; color: #667eea; margin-right: 12px;"></span>
                        <h3 style="margin: 0; color: #2c3e50;">Navigation</h3>
                    </div>
                    <p style="color: #6c757d; margin-bottom: 20px;">Manage your site's navigation menus and structure.</p>
                    <a href="<?php echo admin_url('nav-menus.php'); ?>" class="button button-primary">Manage Navigation</a>
                </div>

                <div class="design-card">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-admin-appearance" style="font-size: 24px; color: #667eea; margin-right: 12px;"></span>
                        <h3 style="margin: 0; color: #2c3e50;">Styles</h3>
                    </div>
                    <p style="color: #6c757d; margin-bottom: 20px;">Customize colors, fonts, and overall styling.</p>
                    <a href="<?php echo admin_url('admin.php?page=elementor'); ?>" class="button button-primary">BuilderPro Styles</a>
                </div>

                <div class="design-card">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-admin-page" style="font-size: 24px; color: #667eea; margin-right: 12px;"></span>
                        <h3 style="margin: 0; color: #2c3e50;">Pages</h3>
                    </div>
                    <p style="color: #6c757d; margin-bottom: 20px;">Create and manage your website pages.</p>
                    <a href="<?php echo admin_url('edit.php?post_type=page'); ?>" class="button button-primary">Manage Pages</a>
                </div>

                <div class="design-card">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-layout" style="font-size: 24px; color: #667eea; margin-right: 12px;"></span>
                        <h3 style="margin: 0; color: #2c3e50;">Templates</h3>
                    </div>
                    <p style="color: #6c757d; margin-bottom: 20px;">Design and manage page templates.</p>
                    <a href="<?php echo admin_url('admin.php?page=elementor-templates'); ?>" class="button button-primary">BuilderPro Templates</a>
                </div>

                <div class="design-card">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-grid-view" style="font-size: 24px; color: #667eea; margin-right: 12px;"></span>
                        <h3 style="margin: 0; color: #2c3e50;">Patterns</h3>
                    </div>
                    <p style="color: #6c757d; margin-bottom: 20px;">Browse and use design patterns.</p>
                    <a href="<?php echo admin_url('admin.php?page=elementor-library'); ?>" class="button button-primary">BuilderPro Patterns</a>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Check if UI should be forced
     *
     * @return bool
     */
    private function shouldForceUI(): bool
    {
        // if (current_user_can('REG121_User')) {
        //     return true;
        // } elseif (defined('SC_FORCE_UI') && SC_FORCE_UI === true) {
        //     return true;
        // } else {
        //     return false;
        // }

        // always force REG121 UI....
        return true;
    }
}
