<?php
/**
 * Modern Admin Menu Enhancement
 *
 * @package _121Digital\Connect
 */

namespace _121Digital\Connect\Admin;

/**
 * ModernAdminMenu Class
 *
 * Enhances the WordPress admin menu with modern styling and functionality
 */
class ModernAdminMenu
{
    /**
     * Register admin menu hooks
     *
     * @return void
     */
    public function register(): void
    {
        add_action('admin_menu', [$this, 'addModernMenuItems'], 999);
        add_action('admin_enqueue_scripts', [$this, 'enqueueModernMenuStyles']);
        add_action('admin_head', [$this, 'addCustomMenuStyles']);
    }

    /**
     * Add modern menu items
     *
     * @return void
     */
    public function addModernMenuItems(): void
    {
        // Add a custom "Design" section similar to the image
        add_menu_page(
            'Design',
            'Design',
            'manage_options',
            'design-overview',
            [$this, 'renderDesignPage'],
            'dashicons-art',
            2
        );

        // Add submenu items under Design
        add_submenu_page(
            'design-overview',
            'Navigation',
            'Navigation',
            'manage_options',
            'design-navigation',
            [$this, 'renderNavigationPage']
        );

        add_submenu_page(
            'design-overview',
            'Styles',
            'Styles',
            'manage_options',
            'design-styles',
            [$this, 'renderStylesPage']
        );

        add_submenu_page(
            'design-overview',
            'Pages',
            'Pages',
            'manage_options',
            'design-pages',
            [$this, 'renderPagesPage']
        );

        add_submenu_page(
            'design-overview',
            'Templates',
            'Templates',
            'manage_options',
            'design-templates',
            [$this, 'renderTemplatesPage']
        );

        add_submenu_page(
            'design-overview',
            'Patterns',
            'Patterns',
            'manage_options',
            'design-patterns',
            [$this, 'renderPatternsPage']
        );
    }

    /**
     * Enqueue modern menu styles
     *
     * @return void
     */
    public function enqueueModernMenuStyles(): void
    {
        if (!$this->shouldForceUI()) {
            return;
        }

        wp_enqueue_style(
            'modern-admin-menu',
            SC_PLUGIN_URL . 'src/Admin/modern-admin-menu.css',
            ['sc-admin-styles'],
            SC_PLUGIN_VERSION
        );
    }

    /**
     * Add custom menu styles
     *
     * @return void
     */
    public function addCustomMenuStyles(): void
    {
        if (!$this->shouldForceUI()) {
            return;
        }
        ?>
        <style>
        /* Simple design section styling */
        #adminmenu .toplevel_page_design-overview .wp-menu-image:before {
            color: #667eea !important;
        }
        </style>
        <?php
    }

    /**
     * Render Design page
     *
     * @return void
     */
    public function renderDesignPage(): void
    {
        ?>
        <div class="wrap">
            <h1>Design</h1>
            <p>Customize the appearance of your website using the block editor.</p>
            
            <div class="design-grid" style="display: grid; grid-template-columns: repeat(auto-fit, minmax(300px, 1fr)); gap: 20px; margin-top: 30px;">
                <div class="design-card" style="background: white; padding: 30px; border-radius: 12px; box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08); border: 1px solid #e9ecef;">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-menu" style="font-size: 24px; color: #667eea; margin-right: 12px;"></span>
                        <h3 style="margin: 0; color: #2c3e50;">Navigation</h3>
                    </div>
                    <p style="color: #6c757d; margin-bottom: 20px;">Manage your site's navigation menus and structure.</p>
                    <a href="<?php echo admin_url('nav-menus.php'); ?>" class="button button-primary">Manage Navigation</a>
                </div>

                <div class="design-card" style="background: white; padding: 30px; border-radius: 12px; box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08); border: 1px solid #e9ecef;">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-admin-appearance" style="font-size: 24px; color: #667eea; margin-right: 12px;"></span>
                        <h3 style="margin: 0; color: #2c3e50;">Styles</h3>
                    </div>
                    <p style="color: #6c757d; margin-bottom: 20px;">Customize colors, fonts, and overall styling.</p>
                    <a href="<?php echo admin_url('admin.php?page=elementor'); ?>" class="button button-primary">BuilderPro Styles</a>
                </div>

                <div class="design-card" style="background: white; padding: 30px; border-radius: 12px; box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08); border: 1px solid #e9ecef;">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-admin-page" style="font-size: 24px; color: #667eea; margin-right: 12px;"></span>
                        <h3 style="margin: 0; color: #2c3e50;">Pages</h3>
                    </div>
                    <p style="color: #6c757d; margin-bottom: 20px;">Create and manage your website pages.</p>
                    <a href="<?php echo admin_url('edit.php?post_type=page'); ?>" class="button button-primary">Manage Pages</a>
                </div>

                <div class="design-card" style="background: white; padding: 30px; border-radius: 12px; box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08); border: 1px solid #e9ecef;">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-layout" style="font-size: 24px; color: #667eea; margin-right: 12px;"></span>
                        <h3 style="margin: 0; color: #2c3e50;">Templates</h3>
                    </div>
                    <p style="color: #6c757d; margin-bottom: 20px;">Design and manage page templates.</p>
                    <a href="<?php echo admin_url('admin.php?page=elementor-templates'); ?>" class="button button-primary">BuilderPro Templates</a>
                </div>

                <div class="design-card" style="background: white; padding: 30px; border-radius: 12px; box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08); border: 1px solid #e9ecef;">
                    <div style="display: flex; align-items: center; margin-bottom: 15px;">
                        <span class="dashicons dashicons-grid-view" style="font-size: 24px; color: #667eea; margin-right: 12px;"></span>
                        <h3 style="margin: 0; color: #2c3e50;">Patterns</h3>
                    </div>
                    <p style="color: #6c757d; margin-bottom: 20px;">Browse and use design patterns.</p>
                    <a href="<?php echo admin_url('admin.php?page=elementor-library'); ?>" class="button button-primary">BuilderPro Patterns</a>
                </div>
            </div>
        </div>
        <?php
    }

    /**
     * Render Navigation page
     *
     * @return void
     */
    public function renderNavigationPage(): void
    {
        wp_redirect(admin_url('nav-menus.php'));
        exit;
    }

    /**
     * Render Styles page
     *
     * @return void
     */
    public function renderStylesPage(): void
    {
        wp_redirect(admin_url('admin.php?page=elementor'));
        exit;
    }

    /**
     * Render Pages page
     *
     * @return void
     */
    public function renderPagesPage(): void
    {
        wp_redirect(admin_url('edit.php?post_type=page'));
        exit;
    }

    /**
     * Render Templates page
     *
     * @return void
     */
    public function renderTemplatesPage(): void
    {
        wp_redirect(admin_url('admin.php?page=elementor-templates'));
        exit;
    }

    /**
     * Render Patterns page
     *
     * @return void
     */
    public function renderPatternsPage(): void
    {
        wp_redirect(admin_url('admin.php?page=elementor-library'));
        exit;
    }

    /**
     * Check if UI should be forced
     *
     * @return bool
     */
    private function shouldForceUI(): bool
    {
        if (current_user_can('REG121_User')) {
            return true;
        } elseif (defined('SC_FORCE_UI') && SC_FORCE_UI === true) {
            return true;
        } else {
            return false;
        }
    }
}
