<?php

namespace _121Digital\Connect\Admin;

use _121Digital\Connect\Admin\Dashboard\Dashboard;
use _121Digital\Connect\Admin\Favicon\Favicon;
use _121Digital\Connect\Admin\Pages\Page;
use _121Digital\Connect\Admin\Pages\Pages;
use _121Digital\Connect\Admin\Posts\Post;
use _121Digital\Connect\Admin\Profile\Profile;
use _121Digital\Connect\Admin\Services\NoticeService;
use _121Digital\Connect\Admin\Settings\Settings;
use _121Digital\Connect\Admin\Styles\AdminInterfaceStyles;
use _121Digital\Connect\Admin\Themes\Themes;

class Admin
{
    /**
     * Admin component instances
     *
     * @var array
     */
    private array $adminComponents = [];

    public function register(): void
    {
        // Initialize and register the NoticeService first, before anything else
        $noticeService = NoticeService::getInstance();
        $noticeService->register();

        add_action('admin_init', [$this, 'maybeRedirectToNetworkAdmin']);
        $this->initializeAdminComponents();
        $this->registerAdminComponents();
    }

    /**
     * Initialize all admin components
     *
     * @return void
     */
    private function initializeAdminComponents(): void
    {
        try {
            $this->adminComponents['admin_menu'] = new AdminMenu();
            $this->adminComponents['admin_bar'] = new AdminBar();
            $this->adminComponents['admin_footer'] = new AdminFooter();
            $this->adminComponents['admin_profile'] = new Profile();
            $this->adminComponents['admin_users'] = new Users();
            $this->adminComponents['admin_updates'] = new AdminUpdates();
            $this->adminComponents['admin_themes'] = new AdminThemes();
            $this->adminComponents['admin_notices'] = new AdminNotices();
            $this->adminComponents['admin_posts'] = new Post();
            $this->adminComponents['dashboard'] = new Dashboard();
            $this->adminComponents['admin_styles'] = new AdminInterfaceStyles();
            $this->adminComponents['contact_requests_view'] = new ContactRequestsView();
            $this->adminComponents['tools'] = new Tools();
            $this->adminComponents['settings'] = new Settings();
            $this->adminComponents['themes'] = new Themes();
            $this->adminComponents['pages'] = new Pages();
            $this->adminComponents['page'] = new Page();
            $this->adminComponents['favicon'] = new Favicon();
            // NoticeService is already registered in the register() method
        } catch (\Exception $e) {
            throw new \Exception('Failed to initialize admin components: ' . $e->getMessage());
        }
    }

    /**
     * Register all admin components
     *
     * @return void
     */
    private function registerAdminComponents(): void
    {
        try {
            foreach ($this->adminComponents as $component) {
                if (method_exists($component, 'register')) {
                    $component->register();
                }
            }
        } catch (\Exception $e) {
            throw new \Exception('Failed to register admin components: ' . $e->getMessage());
        }
    }

    /**
     * Get admin component instance
     *
     * @param string $name
     * @return mixed
     */
    public function getComponent(string $name)
    {
        return $this->adminComponents[$name] ?? null;
    }

    public function maybeRedirectToNetworkAdmin(): void
    {
        // only execute this in admin context
        if (is_user_logged_in() === false || !is_admin()) {
            return;
        }

        // if we are in a multisite node context and user is a network admin,
        // redirect to the network admin dashboard (but only if not already there)
        if (\_121Digital\Connect\Core\Environment::isMultisiteNode() &&
            is_admin() &&
            !is_network_admin() &&
            current_user_can('manage_network')) {
            wp_redirect(network_admin_url());
            exit;
        }
    }
}
